//  GeometryGamesSound.swift
//
//	© 2020 by Jeff Weeks
//	See TermsOfUse.txt

import Foundation
import AVFoundation


//	Play sounds?
var gPlaySounds = true	//	App delegate may override using "sound effects" user pref

//	Thread safety
//
//		AVAudioPlayer's documentation doesn't say whether it's thread-safe or not.
//		But either way, the Geometry Games apps call the following functions
//		from the main thread only.  Given that we're going with single-threaded
//		operation, there's no harm in using the following global variables.
//

//	Each sound cache entry associates an array of AVAudioPlayers to a file name.
//	The file name includes the file extension, for example "Foo.m4a" or "Foo.wav".
var gCachedSoundsForAVAudioPlayer: [String: [AVAudioPlayer]] = [:]

//	Allocate a static buffer to receive the pending sound file name,
//	to avoid having to allocate and initialize such a buffer ≥ 60 times per second
//	in PlayPendingSound().  To be honest, the real advantage is keeping
//	PlayPendingSound()'s code clean -- the performance penalty a run time
//	would likely be negligible.
var gSoundFileNameAsZeroTerminatedString = UnsafeMutablePointer<Char16>.allocate(capacity: Int(SOUND_REQUEST_NAME_BUFFER_LENGTH))

//	Has SetUpAudio() been called?
var gAudioHasBeenSetUp = false


func SetUpAudio() {

	if gAudioHasBeenSetUp {
		return
	}

	//	SetUpAudio() presently sets up an AVAudioSession (which can play M4A), not a MIDI player.
	//	If iOS ever supports MIDI with no need for a custom sound font, we can switch over.

	let theAudioSession = AVAudioSession.sharedInstance()	//	Implicitly initializes the audio session.

	do {
		try theAudioSession.setCategory(AVAudioSession.Category.ambient)	//	Never interrupt background music.
	}
	catch {
		//	Well... I guess we just push on without having set the Category.
	}

	do {
		try theAudioSession.setActive(true)	//	Unnecessary but recommended.
	}
	catch {
		//	Well... I guess we just push on without having set the Active status.
	}
	
	//	An UnsafeMutablePointer's memory must be initialized before use.
	gSoundFileNameAsZeroTerminatedString.initialize(repeating: 0, count: Int(SOUND_REQUEST_NAME_BUFFER_LENGTH))

	//	Setup is complete.
	gAudioHasBeenSetUp = true
}

func ShutDownAudio() {

	//	Never gets called (by design).
	//	But if it did get called, we'd want to stop all sounds
	//	and clear the sound cache.

	ClearSoundCache()
	
	//	Deinitialization isn't needn't for a "trivial type"
	//	(i.e. a type that doesn't allocate further memory of its own),
	//	but I want to get in the habit of deinitializing unsafe pointers,
	//	and in any case the compiler will most likely
	//	optimize away this call to deinitialize().
	//
	gSoundFileNameAsZeroTerminatedString.deinitialize(count: Int(SOUND_REQUEST_NAME_BUFFER_LENGTH))
}

func ClearSoundCache() {

	for theDictionaryEntry in gCachedSoundsForAVAudioPlayer {
		let theAudioPlayerArray = theDictionaryEntry.value
		for theAudioPlayer in theAudioPlayerArray {
			theAudioPlayer.stop()
		}
	}

	gCachedSoundsForAVAudioPlayer.removeAll()
}

func PlayPendingSound() {

	if !gAudioHasBeenSetUp {	//	should never occur
		return
	}

	if (DequeueSoundRequest(gSoundFileNameAsZeroTerminatedString, Int(SOUND_REQUEST_NAME_BUFFER_LENGTH))) {
	
		//	Even if the user has disabled sound effects,
		//	we still want to keep dequeueing sound requests,
		//	but we don't want to play them.
		if ( !gPlaySounds ) {
			return
		}

		//	If background music is playing, don't play the sound.
		if AVAudioSession.sharedInstance().secondaryAudioShouldBeSilencedHint {
			return
		}

		//	Convert the sound file name from a zero-terminated UTF-16 string to a Swift string.
		let theSoundFileName = MakeSwiftString(zeroTerminatedString: gSoundFileNameAsZeroTerminatedString)

		//	Does gCachedSoundsForAVAudioPlayer already contain an array
		//	of AVAudioPlayers for the requested sound?  If not, create one.
		var theAudioPlayerArray = gCachedSoundsForAVAudioPlayer[theSoundFileName] ?? [AVAudioPlayer]()
		
		//	Does theAudioPlayerArray contain an AVAudioPlayer this isn't already playing?
		//	If not, create a new one.
		var thePossibleAudioPlayer: AVAudioPlayer?	//	will soon be guaranteed non-nil
		for theAudioPlayerCandidate in theAudioPlayerArray {
			if !theAudioPlayerCandidate.isPlaying {
				thePossibleAudioPlayer = theAudioPlayerCandidate
				break
			}
		}
		if (thePossibleAudioPlayer == nil)
		{
			guard let theFullPath = Bundle.main.resourcePath?
										.appending("/Sounds - m4a/")
										.appending(theSoundFileName) else {
				return
			}
			let theFileURL = URL(fileURLWithPath: theFullPath, isDirectory: false)
			do {
				let theNewAudioPlayer = try AVAudioPlayer(contentsOf:theFileURL)

				//	Use theNewAudioPlayer now...
				thePossibleAudioPlayer = theNewAudioPlayer

				//	... and cache it for future use.
				//
				//		Caution:  A Swift Array is a struct, not a class,
				//		so it gets passed by value.  Thus we must re-insert
				//		theAudioPlayerArray into the gCachedSoundsForAVAudioPlayer
				//		after we append theNewAudioPlayer, otherwise
				//		theNewAudioPlayer would be lost.
				//
				theAudioPlayerArray.append(theNewAudioPlayer)
				gCachedSoundsForAVAudioPlayer[theSoundFileName] = theAudioPlayerArray
			}
			catch {
				//	Should never occur, unless the file is missing.
				return
			}
		}

		//	Start the sound playing.
		//
		//	There's no need to call theAudioPlayer.prepareToPlay,
		//	given that we'll be calling theAudioPlayer.play immediately anyhow.
		//
		if let theAudioPlayer = thePossibleAudioPlayer {	//	should never fail
			//	gCachedSoundsForAVAudioPlayer keeps a strong reference
			//	to theAudioPlayer, so it's OK to start it playing and return.
			//	(Without that strong reference, the app would crash.)
			theAudioPlayer.play()
		}
	}
}
